import { useState } from 'react';
import { FiSave, FiUpload } from 'react-icons/fi';
import { useNavigate } from 'react-router-dom'; // 1. Import useNavigate

export const EditProfile = () => {
  const navigate = useNavigate(); // 2. Get the navigate function

  // Sample user data - replace with your actual state management
  const [user, setUser] = useState({
    name: "Alex Johnson",
    age: 28,
    bio: "Tech enthusiast who loves hiking and photography",
    profession: "Software Engineer",
    location: "San Francisco, CA",
    interests: ["Hiking", "Photography", "Technology"],
    photos: []
  });

  const [newPhoto, setNewPhoto] = useState(null);

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setUser(prev => ({ ...prev, [name]: value }));
  };

  const handleInterestToggle = (interest) => {
    setUser(prev => ({
      ...prev,
      interests: prev.interests.includes(interest)
        ? prev.interests.filter(i => i !== interest)
        : [...prev.interests, interest]
    }));
  };

  const handlePhotoUpload = (e) => {
    const file = e.target.files[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (event) => {
        setNewPhoto(event.target.result);
      };
      reader.readAsDataURL(file);
    }
  };

  const handleSubmit = (e) => {
    e.preventDefault();
    // Add your save logic here (API call, state update, etc.)
    console.log("Profile updated:", user);

    // 3. Navigate back after saving
    // Use navigate(-1) to go back one step in history
    navigate(-1);
  };

  return (
    <div className="container mx-auto px-4 py-8 max-w-4xl">
      <h1 className="text-3xl font-bold mb-8 text-cyan-400">Edit Profile</h1>

      <form onSubmit={handleSubmit} className="space-y-8">
        {/* Profile Picture Section */}
        <div className="flex flex-col items-center space-y-4">
          <div className="relative w-32 h-32 rounded-full overflow-hidden border-4 border-cyan-400/50">
            <img
              src={newPhoto || "https://randomuser.me/api/portraits/men/32.jpg"}
              alt="Profile"
              className="w-full h-full object-cover"
            />
          </div>
          <label className="px-4 py-2 rounded-lg bg-[rgba(255,255,255,0.1)] border border-[rgba(255,255,255,0.2)] hover:bg-[rgba(255,255,255,0.2)] transition-all cursor-pointer flex items-center space-x-2">
            <FiUpload />
            <span>Upload Photo</span>
            <input
              type="file"
              accept="image/*"
              onChange={handlePhotoUpload}
              className="hidden"
            />
          </label>
        </div>

        {/* Basic Information Section */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-4">
            <h2 className="text-xl font-semibold text-cyan-400">Basic Information</h2>
            <div>
              <label className="block text-sm text-gray-400 mb-1">Full Name</label>
              <input
                type="text"
                name="name"
                value={user.name}
                onChange={handleInputChange}
                className="w-full px-4 py-3 bg-[rgba(255,255,255,0.05)] border border-[rgba(255,255,255,0.1)] rounded-lg focus:outline-none focus:border-cyan-400"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm text-gray-400 mb-1">Age</label>
                <input
                  type="number"
                  name="age"
                  value={user.age}
                  onChange={handleInputChange}
                  className="w-full px-4 py-3 bg-[rgba(255,255,255,0.05)] border border-[rgba(255,255,255,0.1)] rounded-lg focus:outline-none focus:border-cyan-400"
                />
              </div>
              <div>
                <label className="block text-sm text-gray-400 mb-1">Location</label>
                <input
                  type="text"
                  name="location"
                  value={user.location}
                  onChange={handleInputChange}
                  className="w-full px-4 py-3 bg-[rgba(255,255,255,0.05)] border border-[rgba(255,255,255,0.1)] rounded-lg focus:outline-none focus:border-cyan-400"
                />
              </div>
            </div>

            <div>
              <label className="block text-sm text-gray-400 mb-1">Profession</label>
              <input
                type="text"
                name="profession"
                value={user.profession}
                onChange={handleInputChange}
                className="w-full px-4 py-3 bg-[rgba(255,255,255,0.05)] border border-[rgba(255,255,255,0.1)] rounded-lg focus:outline-none focus:border-cyan-400"
              />
            </div>
          </div>

          {/* About Section */}
          <div className="space-y-4">
            <h2 className="text-xl font-semibold text-cyan-400">About You</h2>
            <div>
              <label className="block text-sm text-gray-400 mb-1">Bio</label>
              <textarea
                name="bio"
                value={user.bio}
                onChange={handleInputChange}
                rows="4"
                className="w-full px-4 py-3 bg-[rgba(255,255,255,0.05)] border border-[rgba(255,255,255,0.1)] rounded-lg focus:outline-none focus:border-cyan-400"
              />
            </div>
          </div>
        </div>

        {/* Interests Section */}
        <div>
          <h2 className="text-xl font-semibold text-cyan-400 mb-4">Your Interests</h2>
          <div className="flex flex-wrap gap-3">
            {["Hiking", "Photography", "Technology", "Travel", "Music", "Sports", "Cooking", "Art"].map(interest => (
              <button
                key={interest}
                type="button"
                onClick={() => handleInterestToggle(interest)}
                className={`px-4 py-2 rounded-full text-sm transition-all ${
                  user.interests.includes(interest)
                    ? 'bg-cyan-400/20 text-cyan-400 border border-cyan-400/50'
                    : 'bg-[rgba(255,255,255,0.05)] border border-[rgba(255,255,255,0.1)]'
                }`}
              >
                {interest}
              </button>
            ))}
          </div>
        </div>

        {/* Save Button */}
        <div className="flex justify-end pt-6">
          <button
            type="submit"
            className="px-6 py-3 rounded-lg bg-gradient-to-r from-cyan-400 to-blue-500 text-[#0A0F2A] font-medium hover:shadow-neon-cyan transition-all flex items-center space-x-2"
          >
            <FiSave />
            <span>Save Changes</span>
          </button>
        </div>
      </form>
    </div>
  );
};