import { motion, AnimatePresence } from "framer-motion";
import { useState, useEffect } from "react";
import { Link } from "react-router-dom";
import { useAuth } from "../context/AuthContext.jsx";
import { AuthModal } from "./Auth/AuthModal.jsx";
import { FiMenu, FiX, FiUser, FiEdit, FiHome, FiHeart, FiGrid } from "react-icons/fi"; // Added FiGrid

export const Header = () => {
  const { isLoggedIn, logout, user } = useAuth();
  const [isAuthOpen, setIsAuthOpen] = useState(false);
  const [authTab, setAuthTab] = useState("login");
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

  // Lock body scroll when menu is open
  useEffect(() => {
    if (mobileMenuOpen) {
      document.body.style.overflow = 'hidden';
      document.documentElement.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = '';
      document.documentElement.style.overflow = '';
    }

    return () => {
      document.body.style.overflow = '';
      document.documentElement.style.overflow = '';
    };
  }, [mobileMenuOpen]);

  const toggleMobileMenu = () => {
    setMobileMenuOpen(!mobileMenuOpen);
  };

  return (
    <>
      <header className="sticky top-0 z-50 bg-[rgba(5,10,30,0.9)] border-b border-[rgba(255,255,255,0.1)] backdrop-blur-lg">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <Link to="/" className="text-2xl font-bold text-cyan-400 hover:text-cyan-300 transition-colors">
            Luminous Connections
          </Link>
          
          {/* Desktop Navigation */}
          <nav className="hidden md:flex items-center space-x-6">
            {isLoggedIn ? (
              <motion.div
              initial={{ opacity: 0, x: 20 }}
              animate={{ opacity: 1, x: 0 }}
              className="flex items-center space-x-4"
            >
              {/* Add this Home Link */}
              <Link
                to="/" // Link to the home page
                className="flex items-center space-x-1 text-cyan-400 hover:text-cyan-300 transition-colors"
              >
                <FiHome className="text-sm" /> {/* Home Icon */}
                <span>Home</span>
              </Link>
              {/* Link to Dashboard */}
              <Link
                to="/dashboard"
                className="flex items-center space-x-1 text-cyan-400 hover:text-cyan-300 transition-colors"
              >
                <FiGrid className="text-sm" /> {/* Dashboard Icon */}
                <span>Dashboard</span>
              </Link>
              {/* Existing Links */}
              <Link
                to="/matches"
                className="flex items-center space-x-1 text-cyan-400 hover:text-cyan-300 transition-colors"
              >
                <FiHeart className="text-sm" />
                <span>Matches</span>
              </Link>
              <Link
                to="/profile/edit"
                className="flex items-center space-x-1 text-cyan-400 hover:text-cyan-300 transition-colors"
              >
                <FiEdit className="text-sm" />
                <span>Edit Profile</span>
              </Link>
              {/* ... rest of the logged-in desktop nav */}
            </motion.div>
            ) : (
              <>
                <button 
                  onClick={() => {
                    setAuthTab("login");
                    setIsAuthOpen(true);
                  }}
                  className="px-4 py-2 rounded-lg border border-cyan-400 text-cyan-400 hover:shadow-neon-cyan transition-all"
                >
                  Login
                </button>
                <button 
                  onClick={() => {
                    setAuthTab("signup");
                    setIsAuthOpen(true);
                  }}
                  className="px-4 py-2 rounded-lg bg-gradient-to-r from-cyan-400 to-blue-500 text-[#0A0F2A] hover:shadow-neon-cyan transition-all"
                >
                  Sign Up
                </button>
              </>
            )}
          </nav>

          {/* Mobile Hamburger Button */}
          <button 
            className="md:hidden text-cyan-400 p-2 hover:bg-[rgba(255,255,255,0.1)] rounded-full transition-colors"
            onClick={toggleMobileMenu}
            aria-label="Toggle menu"
          >
            <FiMenu className="w-6 h-6" />
          </button>
        </div>

        {/* Mobile Menu */}
        <AnimatePresence>
          {mobileMenuOpen && (
            <>
              {/* Overlay */}
              <motion.div
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                exit={{ opacity: 0 }}
                className="fixed inset-0 bg-black bg-opacity-80 z-40 md:hidden"
                onClick={toggleMobileMenu}
              />
              
              {/* Mobile Menu Panel */}
              <motion.div
                initial={{ x: '100%' }}
                animate={{ x: 0 }}
                exit={{ x: '100%' }}
                transition={{ type: 'spring', stiffness: 300, damping: 30 }}
                className="fixed top-0 right-0 w-72 h-screen bg-[rgba(5,10,30,0.98)] border-l border-[rgba(255,255,255,0.15)] backdrop-blur-xl z-50 shadow-2xl flex flex-col overflow-hidden"
              >
                {/* Menu Header */}
                <div className="flex-shrink-0 flex justify-between items-center p-6 border-b border-[rgba(255,255,255,0.1)]">
                  <h2 className="text-xl font-bold text-cyan-400">Menu</h2>
                  <button 
                    onClick={toggleMobileMenu}
                    className="text-cyan-400 p-1 hover:bg-[rgba(255,255,255,0.1)] rounded-full transition-colors"
                    aria-label="Close menu"
                  >
                    <FiX className="w-6 h-6" />
                  </button>
                </div>
                
                {/* Scrollable Content */}
                <div className="flex-1 overflow-y-auto no-scrollbar py-2">
                <nav className="px-6 space-y-1">
                <Link
                  to="/"
                  className="flex items-center space-x-3 text-cyan-400 text-lg hover:bg-[rgba(255,255,255,0.05)] p-3 rounded-lg transition-colors"
                  onClick={toggleMobileMenu}
                >
                  <FiHome className="w-5 h-5 flex-shrink-0" />
                  <span>Home</span>
                </Link>

                {isLoggedIn && (
                  <>
                    {/* Link to Dashboard */}
                    <Link
                      to="/dashboard"
                      className="flex items-center space-x-3 text-cyan-400 text-lg hover:bg-[rgba(255,255,255,0.05)] p-3 rounded-lg transition-colors"
                      onClick={toggleMobileMenu}
                    >
                      <FiGrid className="w-5 h-5 flex-shrink-0" /> {/* Dashboard Icon */}
                      <span>Dashboard</span>
                    </Link>
                     {/* Existing Links */}
                    <Link
                      to="/matches"
                      className="flex items-center space-x-3 text-cyan-400 text-lg hover:bg-[rgba(255,255,255,0.05)] p-3 rounded-lg transition-colors"
                      onClick={toggleMobileMenu}
                    >
                      <FiHeart className="w-5 h-5 flex-shrink-0" />
                      <span>My Matches</span>
                    </Link>

                    <Link
                      to="/profile/edit"
                      className="flex items-center space-x-3 text-cyan-400 text-lg hover:bg-[rgba(255,255,255,0.05)] p-3 rounded-lg transition-colors"
                      onClick={toggleMobileMenu}
                    >
                      <FiEdit className="w-5 h-5 flex-shrink-0" />
                      <span>Edit Profile</span>
                    </Link>
                  </>
                )}
              </nav>
                </div>

                {/* User Section */}
                <div className="flex-shrink-0 border-t border-[rgba(255,255,255,0.1)] p-6">
                  {isLoggedIn ? (
                    <div className="flex items-center space-x-3">
                      <div className="w-10 h-10 rounded-full bg-purple-600 flex items-center justify-center flex-shrink-0 border-2 border-cyan-400/50">
                        <FiUser className="text-white text-sm" />
                      </div>
                      <div className="min-w-0">
                        <p className="text-cyan-400 font-medium truncate">{user?.name || "User"}</p>
                        <button 
                          onClick={() => {
                            logout();
                            toggleMobileMenu();
                          }}
                          className="text-xs text-gray-400 hover:text-cyan-400 transition-colors"
                        >
                          Logout
                        </button>
                      </div>
                    </div>
                  ) : (
                    <div className="space-y-3">
                      <button 
                        onClick={() => {
                          setAuthTab("login");
                          setIsAuthOpen(true);
                          toggleMobileMenu();
                        }}
                        className="w-full py-2 rounded-lg border border-cyan-400 text-cyan-400 hover:shadow-neon-cyan transition-all"
                      >
                        Login
                      </button>
                      <button 
                        onClick={() => {
                          setAuthTab("signup");
                          setIsAuthOpen(true);
                          toggleMobileMenu();
                        }}
                        className="w-full py-2 rounded-lg bg-gradient-to-r from-cyan-400 to-blue-500 text-[#0A0F2A] font-medium hover:shadow-neon-cyan transition-all"
                      >
                        Sign Up
                      </button>
                    </div>
                  )}
                </div>
              </motion.div>
            </>
          )}
        </AnimatePresence>
      </header>

      <AuthModal 
        isOpen={isAuthOpen} 
        onClose={() => setIsAuthOpen(false)}
        activeTab={authTab}
        setActiveTab={setAuthTab}
      />
    </>
  );
};