import { AnimatePresence, motion } from "framer-motion";
import { LoginForm } from "./LoginForm";
import { SignupForm } from "./SignupForm";
import { useAuth } from "../../context/AuthContext.jsx";  // Correct import path

export const AuthModal = ({ 
  isOpen, 
  onClose, 
  activeTab,
  setActiveTab 
}) => {
  // Place the hook call at the top level of the component
  const { login } = useAuth();

  // Success handler for both login and signup
  const handleAuthSuccess = () => {
    login();      // Update global auth state
    onClose();    // Close the modal
  };

  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          className="fixed inset-0 z-50 flex items-center justify-center p-4"
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.9 }}
          transition={{ type: "spring", damping: 25 }}
        >
          {/* Backdrop */}
          <div 
            className="absolute inset-0 bg-black bg-opacity-70 backdrop-blur-sm" 
            onClick={onClose} 
          />
          
          {/* Modal Content */}
          <motion.div 
            className="relative w-full max-w-md bg-gradient-to-br from-[rgba(20,25,60,0.8)] to-[rgba(10,15,40,0.9)] border border-[rgba(255,255,255,0.1)] rounded-2xl shadow-2xl overflow-hidden backdrop-blur-lg"
            layout
          >
            {/* Tabs */}
            <div className="flex border-b border-[rgba(255,255,255,0.05)]">
              <button
                className={`flex-1 py-4 font-medium ${activeTab === "login" ? "text-cyan-400" : "text-gray-400"}`}
                onClick={() => setActiveTab("login")}
              >
                Login
              </button>
              <button
                className={`flex-1 py-4 font-medium ${activeTab === "signup" ? "text-cyan-400" : "text-gray-400"}`}
                onClick={() => setActiveTab("signup")}
              >
                Sign Up
              </button>
            </div>
            
            {/* Form Content */}
            <div className="p-6">
              {activeTab === "login" ? (
                <LoginForm 
                  onSuccess={handleAuthSuccess}    // Regular login
                  onDemoLogin={handleAuthSuccess}  // Demo login
                />
              ) : (
                <SignupForm 
                  onSuccess={handleAuthSuccess}    // Regular signup
                  onDemoSignup={handleAuthSuccess} // Demo signup
                />
              )}
            </div>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
};