import { useState } from 'react';
import { motion } from 'framer-motion';
import { ProfileCard } from '../components/ProfileCard';
import { Header } from '../components/Header';
import { CosmicBackground } from '../components/CosmicBackground';
import { demoMatches } from '../data/demoMatches'; // Import demo profiles
import { Footer } from '../components/Footer'; // Add this line

export const Home = () => {
  // 1. Filter State Management
  const [filters, setFilters] = useState({
    ageRange: [18, 40],       // Min/Max age filter
    profession: '',           // Profession search
    location: '',             // Location search
    religion: '',             // Religion filter
    interest: ''              // Interest filter
  });

  // 2. Use demoMatches as initial profiles
  const [filteredProfiles, setFilteredProfiles] = useState(demoMatches);

  // 3. Apply Filters Function
  const applyFilters = () => {
    const results = demoMatches.filter(profile => {
      return (
        // Age filter
        profile.age >= filters.ageRange[0] &&
        profile.age <= filters.ageRange[1] &&
        
        // Profession filter (case insensitive)
        (filters.profession === '' || 
         profile.profession.toLowerCase().includes(filters.profession.toLowerCase())) &&
        
        // Location filter (case insensitive)
        (filters.location === '' || 
         profile.location.toLowerCase().includes(filters.location.toLowerCase())) &&
        
        // Religion exact match
        (filters.religion === '' || profile.religion === filters.religion) &&
        
        // Interest in array
        (filters.interest === '' || 
         profile.interests.some(i => i.toLowerCase().includes(filters.interest.toLowerCase())))
      );
    });
    setFilteredProfiles(results);
  };

  // 4. Handle Filter Changes
  const handleFilterChange = (e) => {
    const { name, value } = e.target;
    setFilters(prev => ({ ...prev, [name]: value }));
  };

  // 5. Handle Age Range Change
  const handleAgeChange = (min, max) => {
    setFilters(prev => ({ ...prev, ageRange: [min, max] }));
  };

  // 6. Reset All Filters
  const resetFilters = () => {
    setFilters({
      ageRange: [18, 40],
      profession: '',
      location: '',
      religion: '',
      interest: ''
    });
    setFilteredProfiles(demoMatches);
  };

  return (
    <div className="min-h-screen">
      {/* Background and Header */}
      <CosmicBackground />
      <Header />
      
      {/* Main Content */}
      <main className="container mx-auto px-4 py-12">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* ===== FILTER SIDEBAR ===== */}
          <motion.div 
            className="lg:w-72 glass-effect p-6 rounded-2xl lg:sticky lg:top-28 lg:h-[calc(100vh-12rem)]"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.5 }}
          >
            <h2 className="text-xl font-bold text-cyan-400 mb-6 flex items-center justify-between">
              <span>Filter Matches</span>
              <button 
                onClick={resetFilters}
                className="text-xs text-gray-400 hover:text-cyan-400"
              >
                Reset All
              </button>
            </h2>
            
            <div className="space-y-6">
              {/* Age Range Filter */}
              <div>
                <label className="block text-sm text-gray-400 mb-2">Age Range</label>
                <div className="flex items-center space-x-2">
                  <input
                    type="number"
                    value={filters.ageRange[0]}
                    onChange={(e) => handleAgeChange(e.target.value, filters.ageRange[1])}
                    className="w-full px-3 py-2 bg-[rgba(5,10,30,0.9)] border border-[rgba(255,255,255,0.2)] rounded-lg text-cyan-400"
                    min="18"
                    max="100"
                  />
                  <span className="text-gray-400">to</span>
                  <input
                    type="number"
                    value={filters.ageRange[1]}
                    onChange={(e) => handleAgeChange(filters.ageRange[0], e.target.value)}
                    className="w-full px-3 py-2 bg-[rgba(5,10,30,0.9)] border border-[rgba(255,255,255,0.2)] rounded-lg text-cyan-400"
                    min="18"
                    max="100"
                  />
                </div>
              </div>

              {/* Profession Filter */}
              <div>
                <label className="block text-sm text-gray-400 mb-2">Profession</label>
                <input
                  type="text"
                  name="profession"
                  value={filters.profession}
                  onChange={handleFilterChange}
                  placeholder="Any profession"
                  className="w-full px-3 py-2 bg-[rgba(5,10,30,0.9)] border border-[rgba(255,255,255,0.2)] rounded-lg text-cyan-400"
                />
              </div>

              {/* Location Filter */}
              <div>
                <label className="block text-sm text-gray-400 mb-2">Location</label>
                <input
                  type="text"
                  name="location"
                  value={filters.location}
                  onChange={handleFilterChange}
                  placeholder="Any location"
                  className="w-full px-3 py-2 bg-[rgba(5,10,30,0.9)] border border-[rgba(255,255,255,0.2)] rounded-lg text-cyan-400"
                />
              </div>

              {/* Religion Filter */}
              <div>
                <label className="block text-sm text-gray-400 mb-2">Religion</label>
                <select
                  name="religion"
                  value={filters.religion}
                  onChange={handleFilterChange}
                  className="w-full px-3 py-2 bg-[rgba(5,10,30,0.9)] border border-[rgba(255,255,255,0.2)] rounded-lg text-cyan-400"
                >
                  <option value="">Any Religion</option>
                  <option value="Hindu" className="bg-[rgba(5,10,30,0.9)]">Hindu</option>
                  <option value="Christian" className="bg-[rgba(5,10,30,0.9)]">Christian</option>
                  <option value="Muslim" className="bg-[rgba(5,10,30,0.9)]">Muslim</option>
                  <option value="Other" className="bg-[rgba(5,10,30,0.9)]">Other</option>
                </select>
              </div>

              {/* Interest Filter */}
              <div>
                <label className="block text-sm text-gray-400 mb-2">Interest</label>
                <input
                  type="text"
                  name="interest"
                  value={filters.interest}
                  onChange={handleFilterChange}
                  placeholder="Any interest"
                  className="w-full px-3 py-2 bg-[rgba(5,10,30,0.9)] border border-[rgba(255,255,255,0.2)] rounded-lg text-cyan-400"
                />
              </div>

              {/* Action Buttons */}
              <div className="flex space-x-3 pt-2">
                <button
                  onClick={applyFilters}
                  className="flex-1 py-2 rounded-lg bg-gradient-to-r from-cyan-400 to-blue-500 text-[#0A0F2A] font-medium hover:shadow-neon-cyan transition-all"
                >
                  Apply
                </button>
              </div>
            </div>
          </motion.div>

          {/* ===== PROFILE RESULTS ===== */}
          <div className="flex-1">
            {/* Results Count */}
            <motion.h2 
              className="text-3xl font-bold text-center mb-8"
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 0.2 }}
            >
              {filteredProfiles.length} {filteredProfiles.length === 1 ? 'Match' : 'Matches'} Found
            </motion.h2>
            
            {/* Profiles Grid */}
            {filteredProfiles.length > 0 ? (
              <motion.div 
                className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                transition={{ delay: 0.4 }}
              >
                {filteredProfiles.map(profile => (
                  <ProfileCard key={profile.id} profile={profile} />
                ))}
              </motion.div>
            ) : (
              <motion.div
                className="text-center py-12"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
              >
                <p className="text-gray-400">No matches found. Try adjusting your filters.</p>
                <button
                  onClick={resetFilters}
                  className="mt-4 px-4 py-2 rounded-lg border border-cyan-400 text-cyan-400 hover:shadow-neon-cyan transition-all"
                >
                  Reset Filters
                </button>
              </motion.div>
            )}
          </div>
        </div>
      </main>

      {/* Footer */}
      <Footer />
    </div>
  );
};