import { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { CosmicBackground } from '../components/CosmicBackground';
import { Header } from '../components/Header';
import { ProfileCard } from '../components/ProfileCard';
import { Link } from 'react-router-dom';
import { demoMatches } from '../data/demoMatches'; // Import demo profiles


export const Dashboard = () => {
  // Mock user data - replace with actual user data from auth context
  const [user, setUser] = useState({
    name: 'Cosmic Explorer',
    age: 28,
    profession: 'Astrophysicist',
    location: 'San Francisco, CA',
    bio: 'Seeking a partner to explore the cosmos with 🚀',
    photo: 'https://randomuser.me/api/portraits/men/32.jpg',
    interests: ['Stargazing', 'Quantum Physics', 'AI'],
    premium: false,
    lastActive: '2 hours ago'
  });

  // Mock data for dashboard
  const [recentMatches, setRecentMatches] = useState([]);
  const [messages, setMessages] = useState([]);
  const [stats, setStats] = useState({
    newMatches: 0,
    messages: 0,
    profileViews: 0
  });
  const [activeTab, setActiveTab] = useState('matches');

  // Fetch data on component mount
  useEffect(() => {
    // In a real app, these would be API calls
    setRecentMatches(demoMatches);

    setMessages([
      { id: 1, sender: 'Stella Nova', preview: 'Hey! Loved your profile...', time: '10m ago', read: false },
      { id: 2, sender: 'Orion Sky', preview: 'What do you think about...', time: '1h ago', read: true },
      { id: 3, sender: 'Luna Star', preview: 'Coffee this weekend?', time: '3h ago', read: true }
    ]);

    setStats({
      newMatches: demoMatches.length,
      messages: 24,
      profileViews: 89
    });
  }, []);

  const upgradeToPremium = () => {
    setUser(prev => ({ ...prev, premium: true }));
    // In real app, would trigger payment flow
  };

  return (
    <div className="min-h-screen bg-[#0A0F2A]">
      <CosmicBackground />
      <Header />

      <main className="container mx-auto px-4 py-8 relative z-10">
        {/* Welcome Section */}
        <motion.div
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
          className="flex flex-col md:flex-row justify-between items-start md:items-center mb-8"
        >
          <div>
            <h1 className="text-3xl font-bold text-cyan-400">Welcome back, {user.name.split(' ')[0]}!</h1>
            <p className="text-gray-400">Last active: {user.lastActive}</p>
          </div>
          {!user.premium && (
            <button
              onClick={upgradeToPremium}
              className="mt-4 md:mt-0 px-6 py-2 rounded-lg bg-gradient-to-r from-cyan-400 to-blue-500 text-[#0A0F2A] font-bold hover:shadow-neon-cyan transition-all"
            >
              Upgrade to Premium
            </button>
          )}
        </motion.div>

        {/* Stats Cards */}
        <motion.div
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ delay: 0.2 }}
          className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8"
        >
          <div className="glass-effect p-6 rounded-2xl">
            <h3 className="text-gray-400 mb-2">New Matches</h3>
            <p className="text-3xl font-bold text-cyan-400">{stats.newMatches}</p>
          </div>
          <div className="glass-effect p-6 rounded-2xl">
            <h3 className="text-gray-400 mb-2">Messages</h3>
            <p className="text-3xl font-bold text-cyan-400">{stats.messages}</p>
          </div>
          <div className="glass-effect p-6 rounded-2xl">
            <h3 className="text-gray-400 mb-2">Profile Views</h3>
            <p className="text-3xl font-bold text-cyan-400">{stats.profileViews}</p>
          </div>
        </motion.div>

        {/* Main Content Tabs */}
        <div className="mb-6 border-b border-[rgba(255,255,255,0.1)]">
          <div className="flex space-x-4">
            <button
              onClick={() => setActiveTab('matches')}
              className={`pb-2 px-1 ${activeTab === 'matches' ? 'border-b-2 border-cyan-400 text-cyan-400' : 'text-gray-400'}`}
            >
              Your Matches
            </button>
            <button
              onClick={() => setActiveTab('messages')}
              className={`pb-2 px-1 ${activeTab === 'messages' ? 'border-b-2 border-cyan-400 text-cyan-400' : 'text-gray-400'}`}
            >
              Messages
            </button>
            <button
              onClick={() => setActiveTab('profile')}
              className={`pb-2 px-1 ${activeTab === 'profile' ? 'border-b-2 border-cyan-400 text-cyan-400' : 'text-gray-400'}`}
            >
              Profile
            </button>
          </div>
        </div>

        {/* Tab Content */}
        {activeTab === 'matches' && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.3 }}
          >
            <h2 className="text-2xl font-bold text-cyan-400 mb-6">Recent Matches</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {recentMatches.map(match => (
                <ProfileCard key={match.id} profile={match} />
              ))}
            </div>
          </motion.div>
        )}

        {activeTab === 'messages' && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.3 }}
            className="glass-effect rounded-2xl overflow-hidden"
          >
            <h2 className="text-2xl font-bold text-cyan-400 mb-6 p-6 pb-0">Your Messages</h2>
            <div className="divide-y divide-[rgba(255,255,255,0.1)]">
              {messages.map(message => (
                <div
                  key={message.id}
                  className={`p-4 hover:bg-[rgba(255,255,255,0.05)] cursor-pointer ${!message.read ? 'bg-[rgba(100,255,255,0.05)]' : ''}`}
                >
                  <div className="flex justify-between items-start">
                    <div className="flex items-center space-x-3">
                      <div className={`h-2 w-2 rounded-full ${!message.read ? 'bg-cyan-400' : 'bg-transparent'}`} />
                      <h3 className="font-bold text-white">{message.sender}</h3>
                    </div>
                    <span className="text-xs text-gray-400">{message.time}</span>
                  </div>
                  <p className="text-gray-400 mt-1 ml-5">{message.preview}</p>
                </div>
              ))}
            </div>
             {messages.length === 0 && (
              <p className="text-gray-400 text-center p-6">No messages yet.</p>
             )}
          </motion.div>
        )}

        {activeTab === 'profile' && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.3 }}
            className="glass-effect p-6 rounded-2xl"
          >
            <h2 className="text-2xl font-bold text-cyan-400 mb-6">Your Profile</h2>
            <div className="flex flex-col md:flex-row gap-6">
              <div className="flex-shrink-0">
                <img
                  src={user.photo}
                  alt={user.name}
                  className="w-32 h-32 rounded-full border-2 border-cyan-400 object-cover"
                />
              </div>
              <div className="flex-grow">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                  <div>
                    <label className="block text-gray-400 text-sm mb-1">Name</label>
                    <p className="text-white">{user.name}</p>
                  </div>
                  <div>
                    <label className="block text-gray-400 text-sm mb-1">Age</label>
                    <p className="text-white">{user.age}</p>
                  </div>
                  <div>
                    <label className="block text-gray-400 text-sm mb-1">Profession</label>
                    <p className="text-white">{user.profession}</p>
                  </div>
                  <div>
                    <label className="block text-gray-400 text-sm mb-1">Location</label>
                    <p className="text-white">{user.location}</p>
                  </div>
                </div>
                
                <div className="mb-6">
                  <label className="block text-gray-400 text-sm mb-1">Bio</label>
                  <p className="text-white">{user.bio}</p>
                </div>
                
                <div>
                  <label className="block text-gray-400 text-sm mb-1">Interests</label>
                  <div className="flex flex-wrap gap-2">
                    {user.interests.map((interest, index) => (
                      <span
                        key={index}
                        className="px-3 py-1 bg-[rgba(5,10,30,0.7)] text-cyan-400 rounded-full text-sm"
                      >
                        {interest}
                      </span>
                    ))}
                  </div>
                </div>
                
                <Link to="/profile/edit">
                  <button className="mt-6 px-6 py-2 rounded-lg bg-gradient-to-r from-cyan-400 to-blue-500 text-[#0A0F2A] font-bold hover:shadow-neon-cyan transition-all">
                    Edit Profile
                  </button>
                </Link>
              </div>
            </div>
          </motion.div>
        )}
      </main>

      {/* Footer */}
      <footer className="glass-effect border-t border-[rgba(255,255,255,0.1)] py-8 mt-12">
        <div className="container mx-auto px-4">
          <div className="flex flex-col md:flex-row justify-between items-center">
            <div className="flex items-center space-x-2 mb-4 md:mb-0">
              <span className="text-cyan-400 font-bold text-xl">Luminous Connections</span>
              <span className="text-gray-400 text-sm">© {new Date().getFullYear()}</span>
            </div>
            <nav className="flex flex-wrap justify-center gap-4 mb-4 md:mb-0">
              <a href="#" className="text-gray-400 hover:text-cyan-400 transition-colors">Privacy</a>
              <a href="#" className="text-gray-400 hover:text-cyan-400 transition-colors">Terms</a>
              <a href="#" className="text-gray-400 hover:text-cyan-400 transition-colors">FAQ</a>
              <a href="#" className="text-gray-400 hover:text-cyan-400 transition-colors">Contact</a>
            </nav>
            <div className="flex space-x-4">
              <a href="#" className="text-gray-400 hover:text-cyan-400 transition-colors">
                <i className="fab fa-twitter"></i>
              </a>
              <a href="#" className="text-gray-400 hover:text-cyan-400 transition-colors">
                <i className="fab fa-instagram"></i>
              </a>
              <a href="#" className="text-gray-400 hover:text-cyan-400 transition-colors">
                <i className="fab fa-facebook"></i>
              </a>
            </div>
          </div>
        </div>
      </footer>
    </div>
  );
};

// NOTE: Added closing brace for the component
// Added imports for Link and Font Awesome icons if needed in footer (assumed via className `fab fa-...`)
// Included ProfileCard component import
// Added placeholder logic for Edit Profile button using Link
// Added empty state for messages tab
// Adjusted match profile structure slightly in useEffect to include photos array for ProfileCard compatibility
// Added example interests to matches for ProfileCard compatibility